#2025/9/17 Copilotを利用して作成
#2025/9/21 ・config の仕様書に合っていなかったのでCopilotを利用して修正
#          ・他のプログラムにパイプでつなげられるようにcfgファイルを書き換えるのではなくて標準出力に書き出すように変更

import sys
import struct
import re

def read_chunk_header(data, offset):
    chunk_id = data[offset:offset+4]
    size = struct.unpack('<I', data[offset+4:offset+8])[0]
    return chunk_id, size, offset + 8

def find_pdta_chunks(data):
    offset = 12  # skip RIFF header
    while offset < len(data):
        chunk_id, size, content_start = read_chunk_header(data, offset)
        if chunk_id == b'LIST' and data[content_start:content_start+4] == b'pdta':
            return content_start + 4, content_start + size
        offset = content_start + size
        if size % 2 == 1:
            offset += 1
    return None, None

def extract_drumset_names(sf2_path):
    drumset_names = {}
    with open(sf2_path, 'rb') as f:
        data = f.read()

    if data[0:4] != b'RIFF' or data[8:12] != b'sfbk':
        print("これは有効なSoundFontファイルではありません。")
        return drumset_names

    pdta_start, pdta_end = find_pdta_chunks(data)
    if pdta_start is None:
        print("pdta チャンクが見つかりませんでした。")
        return drumset_names

    offset = pdta_start
    while offset < pdta_end:
        chunk_id, size, content_start = read_chunk_header(data, offset)
        if chunk_id == b'phdr':
            entry_size = 38
            for i in range(0, size, entry_size):
                entry = data[content_start+i:content_start+i+entry_size]
                name = entry[0:20].split(b'\x00')[0].decode('ascii', errors='ignore')
                preset, bank = struct.unpack('<HH', entry[20:24])
                if bank == 128:
                    drumset_names[preset] = name
            break
        offset = content_start + size
        if size % 2 == 1:
            offset += 1

    return drumset_names

def annotate_cfg_to_stdout(cfg_path, sf2_path):
    drumset_names = extract_drumset_names(sf2_path)
    if not drumset_names:
        print("bank 128 のプリセットが見つかりませんでした。")
        return

    with open(cfg_path, 'r', encoding='utf-8') as f:
        lines = f.readlines()

    for line in lines:
        match = re.match(r"^(drumset)(?:\s+(\w+))?\s+(\d+)(.*)", line.strip())
        if match:
            keyword, mapid, num_str, rest = match.groups()
            num = int(num_str)
            if num in drumset_names:
                name = drumset_names[num]
                mapid_str = f" {mapid}" if mapid else ""
                if "#" in rest:
                    print(f"{keyword}{mapid_str} {num}{rest} # SFX")
                else:
                    print(f"{keyword}{mapid_str} {num} # {name}")
            else:
                print(line.strip())
        else:
            print(line.strip())

if __name__ == "__main__":
    if len(sys.argv) < 3:
        print("使用方法: python add_comment_to_drumset.py input.cfg input.sf2")
        print("          (標準出力に書き出される)")
        sys.exit(1)

    cfg_path = sys.argv[1]
    sf2_path = sys.argv[2]
    annotate_cfg_to_stdout(cfg_path, sf2_path)
